/**
 * Creator for Reader, object responsible for managing feed-sources (it is
 * possible to keep many feed sources) and getting news from feeds
 */
Application.Reader = (function() {

	/**
	 * stores list of active requests so they can be aborted easily
	 * 
	 * @property activeRequests
	 * @type {Array}
	 */
	var activeRequests = [],

	/**
	 * stores all feeds objects
	 * 
	 * @property feeds
	 * @type {Array}
	 */
	feeds = [];

	/**
	 * making request for news from feed
	 * 
	 * @method getFeed
	 * @param {object}
	 *            Feed,
	 * @param {function}
	 *            onResult callback
	 * @private
	 */
	function getFeed(feed, onResult) {
		var req = feed.load(function() {
			requestFinished(req);
			if (onResult)
				onResult(this);
		}, function(err) {
			requestFinished(req); 
			if (onResult)
				onResult(this, err);
		});
		activeRequests.push(req);
		return feed;
	}

	/**
	 * getting news from all feeds or feeds passed as parametr
	 * 
	 * @method getFeeds
	 * @param {Array}
	 *            f - optional list of feeds
	 */
	function getFeeds(f) {
		var fds = f || feeds, feed, foundFeed;
		if (!fds.length) {
			Application.UI.overlay
					.showErrorMsg({
						title : 'No feeds',
						message : "Your list of RSS feeds is empty. Add an RSS channel first."
					});
			return;
		}
		Application.UI.overlay.showLoader({
			title : 'Wait...',
			message : 'Reading RSS channels...1 of ' + fds.length
		});
		for ( var i = 0; i < fds.length; i++) {
			feed = fds[i];
			foundFeed = findFeed(feed);
			if (!foundFeed) {
				feeds.push(feed);
			}
			getFeed(feed, onFeedsResult);
		}
	}

	/**
	 * finding feed on the feed list
	 * 
	 * @method findFeed
	 * @param {object}
	 *            Feed
	 * @return {object} Feed if exist or {object} Null otherwise
	 */
	function findFeed(feed) {
		for ( var i = 0; i < feeds.length; i++) {
			if (feed.url == feeds[i].url) {
				return feeds[i];
			}
		}

		return null;
	}

	/**
	 * callback for getFeeds, invoked after retriving data from one feeds
	 * 
	 * @method onFeedsResult
	 * @private
	 * @param {Array}
	 *            results
	 */
	function onFeedsResult(feed, err) {

		var items = [],
			msg = 'Feed ' + feed.title + ' ';
		if( err ) {
			Application.UI.overlay.loaderMsg(msg + ' could be not loaded. ' + err );
		} else {
			Application.UI.overlay.loaderMsg(msg + ' successfully loaded');
		}
		if (!activeRequests.length) {
			var items = [];
			for ( var i = 0; i < feeds.length; i++) {
				items = items.concat(feeds[i].getItems());
			}
			Application.UI.overlay.hideLoader();
			Application.UI.reader.showItems(items);
		}
	}
	
	/**
	 * invoke when request is finished, remove request from activeRequests
	 * 
	 * @method requestFinished
	 * @param {object} request
	 */
	function requestFinished(request) {

		var index = activeRequests.indexOf(request);
		if (index != -1) {
			activeRequests.splice(index, 1);
		}
	}
	
	/**
	 * Get list of feeds from configuration file
	 */
	function load() {

		var feed;
		if (DEF_FEEDS.length) {
			for ( var i = 0; i < DEF_FEEDS.length; i++) {
				feed = new Feed(DEF_FEEDS[i].url, {color: DEF_FEEDS[i].color});
				feeds.push(feed);
			}
		}
		if (feeds.length) {
			getFeeds(feeds);
		} else {
			Application.UI.overlay.showErrorMsg({
				title : 'No active feed',
				message : 'No source defined.'
			});
		}
	}

	return {
		/**
		 * @method init
		 */
		init : function() {
			Application.UI.bindNav();
			Application.UI.reader.init();
			load();
		},
		/**
		 * display list of news from all feeds
		 */
		showItems : function() {
			var items = [];
			for ( var i = 0; i < feeds.length; i++) {
				items = items.concat(feeds[i].getItems());
			}
			Application.UI.reader.showItems(items);
		},
		/**
		 * 
		 */
		loadItems : function() {
			var x = this;
			if (activeRequests.length)
				x.abort();
			getFeeds();
		},

		/**
		 * aborts all active requests
		 * 
		 * @method abort
		 */
		abort : function() {
			var req;
			while (activeRequests.length) {
				req = activeRequests.pop();
				req.abort();
			}
		}
	};
})();
